<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\GameCatalogController;
use App\Http\Controllers\AdminController;
use Illuminate\Support\Facades\View;
use Illuminate\Http\Request;
use App\Services\Games\ProviderFactory;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use App\Models\Wallet;
use App\Models\Player;

Route::get('/', function () {
    return redirect('/games');
});

Route::get('/register', [AuthController::class, 'showRegister'])->middleware('guest');
Route::post('/register', [AuthController::class, 'register'])->middleware('guest');
Route::get('/login', [AuthController::class, 'showLogin'])->middleware('guest');
Route::post('/login', [AuthController::class, 'login'])->middleware('guest');
Route::post('/logout', [AuthController::class, 'logout'])->middleware('auth');
Route::get('/games', [GameCatalogController::class, 'index']);
Route::get('/docs/games', function (Request $request) {
    $cfg = config('games.providers.agsoftgames');
    $data = [
        'provider' => 'agsoftgames',
        'base_url' => (string)($cfg['base_url'] ?? ''),
        'endpoints' => (array)($cfg['endpoints'] ?? []),
        'callback_url' => rtrim(config('games.callback.base'), '/') . '/api/games/agsoftgames/callback',
        'auth' => [
            'headers' => ['X-MERCHANT-CODE','X-TIMESTAMP','X-NONCE','X-CONTENT-PROCESSING-TYPE','X-SIGN'],
            'sign_type' => 'HmacSHA256',
            'merchant_code' => (string) env('GAMES_MERCHANT_CODE'),
        ],
        'allowed_ips' => (string) env('GAMES_AGSOFT_ALLOWED_IPS'),
        'placeholders' => ['{merchant_code}', '{merchant_name}'],
    ];
    if ($request->expectsJson() || $request->query('format') === 'json') {
        return response()->json($data);
    }
    return View::make('docs.games_en', compact('data'));
});

Route::get('/reports/agsoft', function (Request $request) {
    $client = \App\Services\Games\ProviderFactory::make('agsoftgames');
    $data = [
        'base_url' => config('games.providers.agsoftgames.base_url'),
        'results' => [],
        'logs' => [],
    ];
    if ($client) {
        try { $data['results']['providers'] = $client->providers(); } catch (\Throwable $e) { $data['results']['providers'] = ['status' => 0, 'body' => ['error' => 'exception']]; }
        try { $data['results']['categories'] = $client->categories(); } catch (\Throwable $e) { $data['results']['categories'] = ['status' => 0, 'body' => ['error' => 'exception']]; }
        $catCode = '';
        $catBody = $data['results']['categories']['body'] ?? [];
        if (is_array($catBody)) {
            $list = $catBody['data'] ?? $catBody;
            if (is_array($list)) {
                $first = $list['categoryDataList'][0] ?? $list[0] ?? null;
                if (is_array($first)) { $catCode = $first['categoryCode'] ?? $first['code'] ?? ''; }
            }
        }
        try { $data['results']['games'] = $client->games(['category' => $catCode ?: '']); } catch (\Throwable $e) { $data['results']['games'] = ['status' => 0, 'body' => ['error' => 'exception']]; }
        try { $data['results']['merchant_info'] = $client->merchantInfo(); } catch (\Throwable $e) { $data['results']['merchant_info'] = ['status' => 0, 'body' => ['error' => 'exception']]; }
    }
    $logPath = storage_path('logs/games.log');
    if (\Illuminate\Support\Facades\File::exists($logPath)) {
        $content = \Illuminate\Support\Facades\File::get($logPath);
        $lines = preg_split('/\r?\n/', $content);
        $tail = array_slice($lines, max(count($lines) - 200, 0));
        $data['logs'] = $tail;
    }
    if ($request->expectsJson() || $request->query('format') === 'json') {
        return response()->json($data);
    }
    return \Illuminate\Support\Facades\View::make('reports.agsoft', $data);
});

Route::middleware(['auth', 'role:admin'])->group(function () {
    Route::get('/admin', [AdminController::class, 'index']);
    Route::get('/admin/users', [AdminController::class, 'users']);
    Route::post('/admin/users/{user}/recharge', [AdminController::class, 'recharge']);
    Route::get('/admin/transactions', [AdminController::class, 'transactions']);
});

Route::middleware(['auth'])->group(function () {
    Route::get('/user', function () {
        return \Illuminate\Support\Facades\View::make('user.panel');
    });
    Route::post('/games/{provider}/launch', [\App\Http\Controllers\GameController::class, 'launchRedirect']);
    Route::prefix('wallet')->group(function () {
        Route::get('balance', [\App\Http\Controllers\WalletController::class, 'balance']);
        Route::post('deposit', [\App\Http\Controllers\WalletController::class, 'deposit']);
        Route::post('withdraw', [\App\Http\Controllers\WalletController::class, 'withdraw']);
    });
});

// Public wallet endpoints for Single-Wallet mode (provider callbacks)
Route::prefix('wallet')->withoutMiddleware([\Illuminate\Foundation\Http\Middleware\VerifyCsrfToken::class])->group(function () {
    Route::post('balance', function () {
        $req = request();
        $playerIdRaw = (string) $req->input('playerId', $req->input('player_id', ''));
        $currency = (string) $req->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerIdRaw);
        if (!$player && $playerIdRaw !== '') {
            $player = Player::where('external_id', 'USER-' . $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerIdRaw !== '' ? $playerIdRaw : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        $wallet = Wallet::firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
        return response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format((float) $wallet->balance, 2)]]);
    });
    Route::post('player/info', function () {
        $req = request();
        $playerIdRaw = (string) $req->input('playerId', $req->input('player_id', ''));
        $player = Player::find($playerIdRaw);
        if (!$player && $playerIdRaw !== '') {
            $player = Player::where('external_id', 'USER-' . $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerIdRaw !== '' ? $playerIdRaw : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        $avatar = (string) env('GAMES_PLAYER_AVATAR_URL', '');
        return response()->json([
            'success' => true,
            'code' => 'C10000',
            'msg' => 'Request succeeded',
            'data' => [
                'nickName' => method_exists($player, 'getAttribute') ? ($player->name ?? 'Player') : 'Player',
                'avatarUrl' => $avatar !== '' ? $avatar : '',
            ],
        ]);
    });
    Route::post('bet', function () {
        $req = request();
        $playerIdRaw = (string) $req->input('playerId', $req->input('player_id', ''));
        $amount = (float) $req->input('amount', 0);
        $currency = (string) $req->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerIdRaw);
        if (!$player && $playerIdRaw !== '') {
            $player = Player::where('external_id', 'USER-' . $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerIdRaw !== '' ? $playerIdRaw : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            if ($wallet->balance < $amount) {
                return response()->json(['success' => false, 'code' => 'W40002', 'msg' => 'Balance not enough', 'data' => null]);
            }
            $wallet->decrement('balance', $amount);
            $newBalance = (float) Wallet::where('id', $wallet->id)->value('balance');
            return response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format($newBalance, 2)]]);
        });
    });
    Route::post('win', function () {
        $req = request();
        $playerIdRaw = (string) $req->input('playerId', $req->input('player_id', ''));
        $amount = (float) $req->input('amount', 0);
        $currency = (string) $req->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerIdRaw);
        if (!$player && $playerIdRaw !== '') {
            $player = Player::where('external_id', 'USER-' . $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerIdRaw !== '' ? $playerIdRaw : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            Wallet::where('id', $wallet->id)->update(['balance' => \Illuminate\Support\Facades\DB::raw('balance + ' . $amount)]);
            $newBalance = (float) Wallet::where('id', $wallet->id)->value('balance');
            return response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format($newBalance, 2)]]);
        });
    });
    Route::post('cancel', function () {
        $req = request();
        $playerIdRaw = (string) $req->input('playerId', $req->input('player_id', ''));
        $amount = (float) $req->input('amount', 0);
        $currency = (string) $req->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerIdRaw);
        if (!$player && $playerIdRaw !== '') {
            $player = Player::where('external_id', 'USER-' . $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerIdRaw !== '' ? $playerIdRaw : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            Wallet::where('id', $wallet->id)->update(['balance' => \Illuminate\Support\Facades\DB::raw('balance + ' . $amount)]);
            $newBalance = (float) Wallet::where('id', $wallet->id)->value('balance');
            return response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format($newBalance, 2)]]);
        });
    });
});

Route::get('/games/search', function (Request $request) {
    $q = trim((string) $request->query('q', ''));
    $category = trim((string) $request->query('category', ''));
    $hasImage = in_array(strtolower((string) $request->query('has_image')), ['1','true','yes','on']);
    $gameType = trim((string) $request->query('game_type', ''));
    $provider = 'agsoftgames';
    $query = \App\Models\Game::query()->where('active', true)->where('provider_key', $provider);
    if ($q !== '') {
        $query->where('name', 'like', '%' . str_replace(['%','_'], ['\\%','\\_'], $q) . '%');
    }
    if ($category !== '') {
        $cat = \App\Models\GameCategory::where('name', $category)->first();
        if ($cat) { $query->where('category_id', $cat->id); }
    }
    if ($hasImage) {
        $query->whereNotNull('thumbnail_url');
    }
    if ($gameType !== '') {
        $query->where('meta->game_type', $gameType);
    }
    $items = $query->orderBy('name')->limit(24)->get(['provider_key','code','name','thumbnail_url']);
    return response()->json(['items' => $items]);
});

Route::match(['GET','POST'], '/tools/callback-inspector', function (Request $request) {
    $data = [
        'url' => $request->fullUrl(),
        'method' => $request->method(),
        'ip' => $request->ip(),
        'headers' => $request->headers->all(),
        'query' => $request->query(),
        'payload' => $request->all(),
        'raw' => $request->getContent(),
        'json' => $request->json()->all(),
    ];
    if ($request->expectsJson() || $request->query('format') === 'json') {
        return response()->json($data);
    }
    return View::make('admin.callback_inspector', $data);
})->withoutMiddleware([\Illuminate\Foundation\Http\Middleware\VerifyCsrfToken::class]);
