<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\GameController;
use App\Http\Controllers\GameCallbackController;
use App\Http\Controllers\WalletController;
use App\Http\Middleware\ValidateProviderSignature;
use App\Http\Middleware\ValidateProviderIp;
use App\Services\Games\ProviderFactory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Wallet;
use App\Models\Player;

Route::prefix('games/{provider}')->group(function () {
    Route::post('launch', [GameController::class, 'launch']);
    Route::get('balance', [GameController::class, 'balance']);
    Route::post('debit', [GameController::class, 'debit']);
    Route::post('credit', [GameController::class, 'credit']);
    Route::post('rollback', [GameController::class, 'rollback']);
    Route::post('callback', [GameCallbackController::class, 'handle'])->middleware([ValidateProviderSignature::class]);
});

Route::prefix('agsoft')->group(function () {
    Route::get('providers', function () {
        $client = ProviderFactory::make('agsoftgames');
        $res = $client ? $client->providers() : ['status' => 404, 'body' => ['error' => 'invalid_provider']];
        return response()->json($res['body'] ?? $res, $res['status'] ?? 500);
    });
    Route::get('merchant-info', function () {
        $client = ProviderFactory::make('agsoftgames');
        $res = $client ? $client->merchantInfo() : ['status' => 404, 'body' => ['error' => 'invalid_provider']];
        return response()->json($res['body'] ?? $res, $res['status'] ?? 500);
    });
    Route::get('categories', function () {
        $client = ProviderFactory::make('agsoftgames');
        $res = $client ? $client->categories() : ['status' => 404, 'body' => ['error' => 'invalid_provider']];
        return response()->json($res['body'], $res['status'] ?? 200);
    });
    Route::get('games/{category}', function (string $category) {
        $client = ProviderFactory::make('agsoftgames');
        $res = $client ? $client->games(['category' => $category]) : ['status' => 404, 'body' => ['error' => 'invalid_provider']];
        return response()->json($res['body'], $res['status'] ?? 200);
    });
    Route::get('game-url/{gameCode}', function (string $gameCode) {
        $client = ProviderFactory::make('agsoftgames');
        $userId = auth()->id() ?? 'GUEST_' . rand(10000, 99999);
        $res = $client ? $client->launch(['player_id' => (string) $userId, 'game_code' => $gameCode, 'currency' => env('GAMES_DEFAULT_CURRENCY', 'BD_BDT')]) : ['status' => 404, 'body' => ['error' => 'invalid_provider']];
        return response()->json($res['body'], $res['status'] ?? 200);
    });
    Route::get('merchant-info', function () {
        $client = ProviderFactory::make('agsoftgames');
        $res = $client ? $client->merchantInfo() : ['status' => 404, 'body' => ['error' => 'invalid_provider']];
        return response()->json($res['body'], $res['status'] ?? 200);
    });
});

Route::prefix('games/agsoftgames/callback')->middleware([ValidateProviderSignature::class])->group(function () {
    Route::post('wallet/balance', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.balance.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
        ]);
        $payload = $req->all();
        $playerIdRaw = (string) ($payload['playerId'] ?? $payload['player_id'] ?? $req->query('playerId', $req->query('player_id', '')));
        $playerKey = preg_replace('/^player_/i', '', $playerIdRaw);
        $currency = (string) env('GAMES_DEFAULT_CURRENCY', 'BD_BDT');
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        $wallet = Wallet::firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
        $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format((float) $wallet->balance, 2, '.', '')]]);
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.balance.response', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'status' => $response->getStatusCode(),
            'body' => $response->getContent(),
            'headers' => $response->headers->all(),
            'event' => 'merchant_wallet_balance_response',
        ]);
        return $response;
    });
    Route::post('player/info', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.player.info.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
        ]);
        $payload = $req->all();
        $playerIdRaw = (string) ($payload['playerId'] ?? $payload['player_id'] ?? '');
        $playerKey = preg_replace('/^player_/i', '', $playerIdRaw);
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        $avatar = (string) env('GAMES_PLAYER_AVATAR_URL', '');
        $response = response()->json([
            'success' => true,
            'code' => 'C10000',
            'msg' => 'Request succeeded',
            'data' => [
                'nickName' => method_exists($player, 'getAttribute') ? ($player->name ?? 'Player') : 'Player',
                'avatarUrl' => $avatar !== '' ? $avatar : '',
            ],
        ]);
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.player.info.response', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'status' => $response->getStatusCode(),
            'body' => $response->getContent(),
            'headers' => $response->headers->all(),
            'event' => 'merchant_player_info_response',
        ]);
        return $response;
    });
    Route::post('wallet/bet', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.bet.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
        ]);
        $payload = $req->all();
        $playerIdRaw = (string) ($payload['playerId'] ?? $payload['player_id'] ?? '');
        $playerKey = preg_replace('/^player_/i', '', $playerIdRaw);
        $amount = (float) ($payload['betAmount'] ?? $payload['amount'] ?? 0);
        $currency = (string) ($payload['currencyCode'] ?? env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency, $payload, $req) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            if ($wallet->balance < $amount) {
                $response = response()->json(['success' => false, 'code' => 'W40002', 'msg' => 'Balance not enough', 'data' => null]);
                \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.bet.response', [
                    'method' => $req->method(),
                    'ip' => $req->ip(),
                    'status' => $response->getStatusCode(),
                    'body' => $response->getContent(),
                    'headers' => $response->headers->all(),
                    'event' => 'merchant_wallet_bet_response',
                ]);
                return $response;
            }
            $wallet->decrement('balance', $amount);
            $newBalance = (float) Wallet::where('id', $wallet->id)->value('balance');
            $mbet = 'MBET_' . ((string) ($payload['betId'] ?? $payload['transactionId'] ?? bin2hex(random_bytes(4))));
            $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['merchantBetId' => $mbet, 'balance' => number_format($newBalance, 2, '.', '')]]);
            \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.bet.response', [
                'method' => $req->method(),
                'ip' => $req->ip(),
                'status' => $response->getStatusCode(),
                'body' => $response->getContent(),
                'headers' => $response->headers->all(),
                'event' => 'merchant_wallet_bet_response',
            ]);
            return $response;
        });
    });
    Route::post('wallet/win', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.win.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
        ]);
        $payload = $req->all();
        $playerIdRaw = (string) ($payload['playerId'] ?? $payload['player_id'] ?? '');
        $playerKey = preg_replace('/^player_/i', '', $playerIdRaw);
        $betAmount = (float) ($payload['betAmount'] ?? 0);
        $winAmount = (float) ($payload['winAmount'] ?? $payload['amount'] ?? 0);
        $type = (string) ($payload['type'] ?? '');
        $currency = (string) ($payload['currencyCode'] ?? env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $betAmount, $winAmount, $type, $currency, $payload, $req) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            if ($type === 'bet_win' && $betAmount > 0) {
                if ($wallet->balance < $betAmount) {
                    $response = response()->json(['success' => false, 'code' => 'W40002', 'msg' => 'Balance not enough', 'data' => null]);
                    \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.win.response', [
                        'method' => $req->method(),
                        'ip' => $req->ip(),
                        'status' => $response->getStatusCode(),
                        'body' => $response->getContent(),
                        'headers' => $response->headers->all(),
                        'event' => 'merchant_wallet_win_response',
                    ]);
                    return $response;
                }
                $wallet->decrement('balance', $betAmount);
            }
            if ($winAmount > 0) {
                $wallet->increment('balance', $winAmount);
            }
            $newBalance = (float) Wallet::where('id', $wallet->id)->value('balance');
            $mbet = 'MBET_' . ((string) ($payload['betId'] ?? $payload['transactionId'] ?? bin2hex(random_bytes(4))));
            $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['merchantBetId' => $mbet, 'balance' => number_format($newBalance, 2, '.', '')]]);
            \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.win.response', [
                'method' => $req->method(),
                'ip' => $req->ip(),
                'status' => $response->getStatusCode(),
                'body' => $response->getContent(),
                'headers' => $response->headers->all(),
                'event' => 'merchant_wallet_win_response',
            ]);
            return $response;
        });
    });
    Route::post('wallet/cancel', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.cancel.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
        ]);
        $payload = $req->all();
        $playerIdRaw = (string) ($payload['playerId'] ?? $payload['player_id'] ?? '');
        $playerKey = preg_replace('/^PLAYER_/', '', $playerIdRaw);
        $amount = (float) ($payload['amount'] ?? 0);
        $currency = (string) env('GAMES_DEFAULT_CURRENCY', 'BD_BDT');
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency, $req) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            if ($amount > 0) {
                $wallet->increment('balance', $amount);
            }
            $newBalance = (float) Wallet::where('id', $wallet->id)->value('balance');
            $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format($newBalance, 2, '.', '')]]);
            \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.merchant.wallet.cancel.response', [
                'method' => $req->method(),
                'ip' => $req->ip(),
                'status' => $response->getStatusCode(),
                'body' => $response->getContent(),
                'headers' => $response->headers->all(),
                'event' => 'merchant_wallet_cancel_response',
            ]);
            return $response;
        });
    });
});
Route::prefix('agsoft/callback')->group(function () {

    Route::match(['GET','POST'],'balance', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.balance.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
            'event' => 'agsoft_wallet_balance_request',
        ]);
        $payload = $req->all();
        $playerId = (string) ($payload['playerId'] ?? $req->query('playerId', ''));
        $playerKey = preg_replace('/^player_/i', '', $playerId);
        $currency = (string) env('GAMES_DEFAULT_CURRENCY', 'BD_BDT');
        $player = Player::find($playerKey);
        if (!$player && $playerKey !== '') {
            $player = Player::where('external_id', 'USER-' . $playerKey)->first();
        }
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        $wallet = Wallet::firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
        $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format((float) $wallet->balance, 2, '.', '')]]);
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.balance.response', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'status' => $response->getStatusCode(),
            'body' => $response->getContent(),
            'headers' => $response->headers->all(),
            'event' => 'agsoft_wallet_balance_response',
        ]);
        return $response;
    });
    Route::post('bet', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.bet.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
            'event' => 'agsoft_wallet_bet_request',
        ]);
        $payload = $req->all();
        $playerId = (string) ($payload['playerId'] ?? '');
        $playerKey = preg_replace('/^player_/i', '', $playerId);
        $amount = (float) ($payload['amount'] ?? 0);
        $currency = (string) env('GAMES_DEFAULT_CURRENCY', 'BD_BDT');
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency, $req) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            if ($wallet->balance < $amount) {
                $response = response()->json(['success' => false, 'code' => 'W40002', 'msg' => 'Balance not enough', 'data' => null]);
                \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.bet.response', [
                    'method' => $req->method(),
                    'ip' => $req->ip(),
                    'status' => $response->getStatusCode(),
                    'body' => $response->getContent(),
                    'headers' => $response->headers->all(),
                    'event' => 'agsoft_wallet_bet_response',
                ]);
                return $response;
            }
            $wallet->balance = $wallet->balance - $amount;
            $wallet->save();
            $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format((float) $wallet->balance, 2, '.', '')]]);
            \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.bet.response', [
                'method' => $req->method(),
                'ip' => $req->ip(),
                'status' => $response->getStatusCode(),
                'body' => $response->getContent(),
                'headers' => $response->headers->all(),
                'event' => 'agsoft_wallet_bet_response',
            ]);
            return $response;
        });
    });
    Route::post('win', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.win.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
            'event' => 'agsoft_wallet_win_request',
        ]);
        $payload = $req->all();
        $playerId = (string) ($payload['playerId'] ?? '');
        $playerKey = preg_replace('/^player_/i', '', $playerId);
        $amount = (float) ($payload['amount'] ?? 0);
        $currency = (string) env('GAMES_DEFAULT_CURRENCY', 'BD_BDT');
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency, $req) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            $wallet->balance = $wallet->balance + $amount;
            $wallet->save();
            $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded', 'data' => ['balance' => number_format((float) $wallet->balance, 2, '.', '')]]);
            \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.win.response', [
                'method' => $req->method(),
                'ip' => $req->ip(),
                'status' => $response->getStatusCode(),
                'body' => $response->getContent(),
                'headers' => $response->headers->all(),
                'event' => 'agsoft_wallet_win_response',
            ]);
            return $response;
        });
    });
    Route::post('cancel', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.cancel.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
            'event' => 'agsoft_wallet_cancel_request',
        ]);
        $payload = $req->all();
        $playerId = (string) ($payload['playerId'] ?? '');
        $playerKey = preg_replace('/^player_/i', '', $playerId);
        $amount = (float) ($payload['amount'] ?? 0);
        $currency = (string) ($payload['currencyCode'] ?? env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerKey) ?: Player::where('external_id', 'USER-' . $playerKey)->first();
        if (!$player) {
            $player = Player::create(['external_id' => 'USER-' . ($playerKey !== '' ? $playerKey : ('GUEST_' . rand(10000, 99999))), 'status' => 'ACTIVE']);
        }
        return DB::transaction(function () use ($player, $amount, $currency, $req) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $player->id, 'currency' => $currency]);
            $wallet->balance = $wallet->balance + $amount;
            $wallet->save();
            $response = response()->json(['success' => true, 'code' => 'C10000', 'msg' => 'Request succeeded']);
            \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.cancel.response', [
                'method' => $req->method(),
                'ip' => $req->ip(),
                'status' => $response->getStatusCode(),
                'body' => $response->getContent(),
                'headers' => $response->headers->all(),
                'event' => 'agsoft_wallet_cancel_response',
            ]);
            return $response;
        });
    });
    Route::post('player-info', function () {
        $req = request();
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.player_info.request', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'query' => $req->query(),
            'payload' => $req->all(),
            'headers' => $req->headers->all(),
            'raw' => $req->getContent(),
            'event' => 'agsoft_wallet_player_info_request',
        ]);
        $playerId = (string) $req->input('playerId');
        $player = Player::find($playerId);
        $response = response()->json(['code' => 'C10000', 'data' => [
            'playerId' => $player->id ?? $playerId,
            'nickName' => method_exists($player, 'getAttribute') ? ($player->name ?? 'Player') : 'Player',
            'status' => 'ACTIVE',
        ]]);
        \Illuminate\Support\Facades\Log::channel('games')->info('agsoft.callback.player_info.response', [
            'method' => $req->method(),
            'ip' => $req->ip(),
            'status' => $response->getStatusCode(),
            'body' => $response->getContent(),
            'headers' => $response->headers->all(),
            'event' => 'agsoft_wallet_player_info_response',
        ]);
        return $response;
    });
});
