<?php

namespace App\Services\Games\Providers;

use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Facades\Log;

class AgulaHoccyClient
{
    protected string $baseUrl;
    protected ?string $apiKey;
    protected ?string $secret;
    protected array $endpoints;
    protected int $timeout;
    protected array $attempts = [];

    public function __construct(array $config)
    {
        $this->baseUrl = rtrim((string)($config['base_url'] ?? ''), '/');
        $this->apiKey = $config['api_key'] ?? null;
        $this->secret = $config['secret'] ?? null;
        $this->endpoints = $config['endpoints'] ?? [];
        $this->timeout = (int)($config['timeout'] ?? 10);
    }

    public function launch(array $payload): array
    {
        // $country = (string) env('GAMES_COUNTRY_CODE', 'BR');
        // $mode = (string) env('GAMES_GAME_MODE', 'REAL');
        $avatar = (string) env('GAMES_PLAYER_AVATAR_URL', '');
        $limitsJson = (string) env('GAMES_BET_LIMITS_JSON', '');
        if ($limitsJson !== '') {
            try { $decoded = json_decode($limitsJson, true); if (is_array($decoded)) { $limits = $decoded; } } catch (\Throwable $e) {}
        }
        $pid = (string)($payload['player_id'] ?? '');
        $playerId = $pid !== '' && preg_match('/^player_/i', $pid) ? $pid : ($pid !== '' ? ('player_' . $pid) : '');
        $mapped = [
            'reqTraceId' => 'TRACE_' . bin2hex(random_bytes(4)),
            'gameCode' => (string)($payload['game_code'] ?? ''),
            'playerId' => $playerId,
            'currencyCode' => "BD_BDT",
            'language' => (string) env('GAMES_DEFAULT_LANG', 'en'),
            'terminalType' => 'PHONE',
            'returnUrl' => (string) (env('GAMES_RETURN_URL') ?: (rtrim((string) config('app.url'), '/') . '/games')),
            'ipAddress' => request()->ip(),
            'subMerchantCode' => (string) env('GAMES_SUB_MERCHANT_CODE', 'SITE01'),
            'nickName' => auth()->user()->name ?? 'Player',
            'avatarUrl' => $avatar ?: null,
            'countryCode' => "BD",
        ];
     
        $mapped = array_filter($mapped, function ($v) { return $v !== null; });
        return $this->request('launch', $mapped, 'POST');
    }

    public function balance(array $payload): array
    {
        return $this->request('balance', $payload, 'GET');
    }

    public function debit(array $payload): array
    {
        return $this->request('debit', $payload);
    }

    public function credit(array $payload): array
    {
        return $this->request('credit', $payload);
    }

    public function rollback(array $payload): array
    {
        return $this->request('rollback', $payload);
    }

    public function providers(): array
    {
        $payload = ['reqTraceId' => 'TRACE_' . bin2hex(random_bytes(4))];
        return $this->request('providers', $payload, 'POST');
    }

    public function categories(string $currencyCode = null, string $lang = null): array
    {
        $payload = [
            'reqTraceId' => 'TRACE_' . bin2hex(random_bytes(4)),
            'currencyCode' => (string)($currencyCode ?? env('GAMES_DEFAULT_CURRENCY', 'BD_BDT')),
            'lang' => (string)($lang ?? env('GAMES_DEFAULT_LANG', 'pt')),
        ];
        return $this->request('categories', $payload, 'POST');
    }

    public function merchantInfo(): array
    {
        $payload = ['reqTraceId' => 'TRACE_' . bin2hex(random_bytes(4))];
        return $this->request('merchant_info', $payload, 'POST');
    }

    public function games(array $params = []): array
    {
        $this->attempts = [];
        $mapped = [
            'reqTraceId' => 'TRACE_' . bin2hex(random_bytes(4)),
            'categoryCode' => (string)($params['category'] ?? ''),
            'currencyCode' => (string) env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'),
            'lang' => (string) env('GAMES_DEFAULT_LANG', 'pt'),
            'pageIndex' => (int)($params['page'] ?? 1),
            'pageSize' => (int)($params['pageSize'] ?? 50),
        ];
        $res = $this->request('games', $mapped, 'POST');
        if (($res['status'] ?? 0) === 200) {
            return $res;
        }
        $candidates = (array) ($this->endpoints['games_candidates'] ?? []);
        foreach ($candidates as $path) {
            $try = $this->requestPath($path, $mapped, 'POST');
            if (($try['status'] ?? 0) === 200) {
                return $try;
            }
        }
        return $res;
    }

    protected function request(string $key, array $payload, string $method = 'POST'): array
    {
        $path = $this->formatPath($this->endpoints[$key] ?? '');
        $url = $this->baseUrl . $path;

        $client = Http::timeout($this->timeout)->retry(3, 200)->acceptJson();

        $bodyJson = $method === 'GET' ? '' : json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $headers = $this->buildHmacHeaders($bodyJson);
        $client = $client->withHeaders($headers);
        try {
            $response = $method === 'GET'
                ? $client->get($url)
                : $client->withBody($bodyJson, 'application/json')->post($url);

            Log::channel('games')->info('provider.request', [
                'url' => $url,
                'method' => $method,
                'status' => $response->status(),
                'body' => $response->body(),
                'event' => 'provider_request',
            ]);
            $this->attempts[] = ['url' => $url, 'method' => $method, 'status' => $response->status()];

            return [
                'status' => $response->status(),
                'body' => $response->json() ?? $response->body(),
                'headers' => $response->headers(),
            ];
        } catch (RequestException $e) {
            Log::channel('games')->warning('provider.request_error', [
                'url' => $url,
                'method' => $method,
                'status' => $e->response ? $e->response->status() : 0,
                'error' => $e->getMessage(),
                'body' => $e->response ? $e->response->body() : null,
                'event' => 'provider_request_error',
            ]);
            $this->attempts[] = ['url' => $url, 'method' => $method, 'status' => $e->response ? $e->response->status() : 0];
            return [
                'status' => $e->response ? $e->response->status() : 0,
                'body' => ['error' => 'provider_request_failed'],
                'headers' => $e->response ? $e->response->headers() : [],
            ];
        } catch (\Throwable $e) {
            Log::channel('games')->error('provider.request_exception', [
                'url' => $url,
                'method' => $method,
                'error' => $e->getMessage(),
                'event' => 'provider_request_exception',
            ]);
            $this->attempts[] = ['url' => $url, 'method' => $method, 'status' => 0];
            return [
                'status' => 0,
                'body' => ['error' => 'provider_unavailable'],
                'headers' => [],
            ];
        }
    }

    protected function requestPath(string $path, array $payload, string $method = 'POST'): array
    {
        $url = $this->baseUrl . $this->formatPath($path);
        $client = Http::timeout($this->timeout)->retry(3, 200)->acceptJson();
        $bodyJson = $method === 'GET' ? '' : json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $headers = $this->buildHmacHeaders($bodyJson);
        $client = $client->withHeaders($headers);
        try {
            $response = $method === 'GET' ? $client->get($url) : $client->withBody($bodyJson, 'application/json')->post($url);
            Log::channel('games')->info('provider.request', [
                'url' => $url,
                'method' => $method,
                'status' => $response->status(),
            ]);
            $this->attempts[] = ['url' => $url, 'method' => $method, 'status' => $response->status()];
            return [
                'status' => $response->status(),
                'body' => $response->json() ?? $response->body(),
                'headers' => $response->headers(),
            ];
        } catch (RequestException $e) {
            Log::channel('games')->warning('provider.request_error', [
                'url' => $url,
                'method' => $method,
                'status' => $e->response ? $e->response->status() : 0,
            ]);
            $this->attempts[] = ['url' => $url, 'method' => $method, 'status' => $e->response ? $e->response->status() : 0];
            return [
                'status' => $e->response ? $e->response->status() : 0,
                'body' => ['error' => 'provider_request_failed'],
                'headers' => $e->response ? $e->response->headers() : [],
            ];
        } catch (\Throwable $e) {
            Log::channel('games')->error('provider.request_exception', [
                'url' => $url,
                'method' => $method,
            ]);
            $this->attempts[] = ['url' => $url, 'method' => $method, 'status' => 0];
            return [
                'status' => 0,
                'body' => ['error' => 'provider_unavailable'],
                'headers' => [],
            ];
        }
    }

    public function attempts(): array
    {
        return $this->attempts;
    }

    protected function formatPath(string $path): string
    {
        $tokens = [
            '{merchant_code}' => (string) (env('GAMES_MERCHANT_CODE') ?: config('services.agsoft.merchant_code')),
            '{merchant_name}' => (string) (env('GAMES_MERCHANT_NAME') ?: (config('services.agsoft.merchant_code') ? 'AGSOFT' : '')),
        ];
        return strtr($path, $tokens);
    }

    public function verifySignature(string $payload, ?string $signature, array $headers = []): bool
    {
        if (!$this->secret || !$signature) {
            return false;
        }
        $expected = hash_hmac('sha256', $payload, $this->secret);
        if (hash_equals($expected, $signature)) {
            return true;
        }
        $mc = $headers['x-merchant-code'][0] ?? $headers['X-MERCHANT-CODE'][0] ?? null;
        $ts = $headers['x-timestamp'][0] ?? $headers['X-TIMESTAMP'][0] ?? null;
        $nonce = $headers['x-nonce'][0] ?? $headers['X-NONCE'][0] ?? null;
        $stype = $headers['x-content-processing-type'][0] ?? $headers['X-CONTENT-PROCESSING-TYPE'][0] ?? null;
        if ($mc && $ts && $nonce && $stype) {
            $signString = (string) $mc . (string) $ts . (string) $nonce . (string) $stype . (string) $payload;
            $expected2 = hash_hmac('sha256', $signString, (string) $this->secret);
            if (hash_equals($expected2, $signature)) {
                return true;
            }
        }
        return false;
    }

    protected function buildHmacHeaders(string $bodyJson): array
    {
        $merchantCode = (string) (env('GAMES_MERCHANT_CODE') ?: config('services.agsoft.merchant_code'));
        $signType = 'HmacSHA256';
        $timestamp = (string) (int) round(microtime(true) * 1000);
        $nonce = bin2hex(random_bytes(8));
        $signString = $merchantCode . $timestamp . $nonce . $signType . $bodyJson;
        $signature = hash_hmac('sha256', $signString, (string) $this->secret);
        return [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'X-MERCHANT-CODE' => $merchantCode,
            'X-TIMESTAMP' => $timestamp,
            'X-NONCE' => $nonce,
            'X-CONTENT-PROCESSING-TYPE' => $signType,
            'X-SIGN' => $signature,
        ];
    }
}

