<?php

namespace App\Http\Controllers;

use App\Services\Games\ProviderFactory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\Player;
use App\Models\Wallet;

class GameController extends Controller
{
    public function launchRedirect(Request $request, string $provider)
    {
        $request->merge([
            'player_id' => $request->input('player_id', $request->input('playerId')),
            'game_code' => $request->input('game_code', $request->input('gameCode')),
            'currency' => $request->input('currency', $request->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'))),
        ]);
        $data = $request->validate([
            'player_id' => 'required|string',
            'game_code' => 'required|string',
            'currency' => 'required|string',
        ]);
        $playerIdRaw = (string)($data['player_id'] ?? '');
        $currency = (string)($data['currency'] ?? env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerIdRaw);
        if (!$player) {
            $player = Player::where('user_id', $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create([
                'user_id' => is_numeric($playerIdRaw) ? (int)$playerIdRaw : null,
                'external_id' => 'USER-' . $playerIdRaw,
                'status' => 'ACTIVE',
            ]);
        }
        Wallet::firstOrCreate(['player_id' => $player->id, 'currency' => $currency], ['balance' => 0]);
        $data['player_id'] = (string) $player->id;
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->view('errors.404', [], 404);
        }
        $res = $client->launch($data);
        $body = $res['body'] ?? [];
        $url = null;
        if (is_array($body)) {
            $dataPart = $body['data'] ?? [];
            if (is_array($dataPart)) {
                $url = (string)($dataPart['url'] ?? $dataPart['gameUrl'] ?? $dataPart['launchUrl'] ?? '');
            } else {
                $url = (string)($body['url'] ?? $body['gameUrl'] ?? $body['launchUrl'] ?? '');
            }
        } elseif (is_string($body)) {
            $url = $body;
        }
        if ($url && strpos($url, 'http') === 0) {
            return redirect()->away($url);
        }
        return response()->json($body, (int)($res['status'] ?? 500));
    }
    public function launch(Request $request, string $provider): JsonResponse
    {
        $request->merge([
            'player_id' => $request->input('player_id', $request->input('playerId')),
            'game_code' => $request->input('game_code', $request->input('gameCode')),
            'currency' => $request->input('currency', $request->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'))),
        ]);
        $data = $request->validate([
            'player_id' => 'required|string',
            'game_code' => 'required|string',
            'currency' => 'required|string',
        ]);
        $playerIdRaw = (string)($data['player_id'] ?? '');
        $currency = (string)($data['currency'] ?? env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'));
        $player = Player::find($playerIdRaw);
        if (!$player) {
            $player = Player::where('user_id', $playerIdRaw)->first();
        }
        if (!$player) {
            $player = Player::create([
                'user_id' => is_numeric($playerIdRaw) ? (int)$playerIdRaw : null,
                'external_id' => 'USER-' . $playerIdRaw,
                'status' => 'ACTIVE',
            ]);
        }
        Wallet::firstOrCreate(['player_id' => $player->id, 'currency' => $currency], ['balance' => 0]);
        $data['player_id'] = (string) $player->id;
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->json(['error' => 'invalid_provider'], 404);
        }
        $res = $client->launch($data);
        return response()->json($res['body'], $res['status']);
    }

    public function balance(Request $request, string $provider): JsonResponse
    {
        \Illuminate\Support\Facades\Log::channel('games')->info('games.balance.request', [
            'provider' => $provider,
            'method' => $request->method(),
            'ip' => $request->ip(),
            'query' => $request->query(),
            'headers' => $request->headers->all(),
        ]);
        $request->merge([
            'player_id' => $request->input('player_id', $request->input('playerId')),
            'currency' => $request->input('currency', $request->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'))),
        ]);
        $data = $request->validate([
            'player_id' => 'required|string',
            'currency' => 'required|string',
        ]);
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->json(['error' => 'invalid_provider'], 404);
        }
        $res = $client->balance($data);
        \Illuminate\Support\Facades\Log::channel('games')->info('games.balance.response', [
            'provider' => $provider,
            'status' => $res['status'] ?? null,
            'headers' => $res['headers'] ?? [],
            'body' => $res['body'] ?? null,
        ]);
        return response()->json($res['body'], $res['status']);
    }

    public function debit(Request $request, string $provider): JsonResponse
    {
        $request->merge([
            'player_id' => $request->input('player_id', $request->input('playerId')),
            'currency' => $request->input('currency', $request->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'))),
            'tx_id' => $request->input('tx_id', $request->input('txId')),
        ]);
        $data = $request->validate([
            'player_id' => 'required|string',
            'amount' => 'required|numeric',
            'currency' => 'required|string',
            'tx_id' => 'required|string',
        ]);
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->json(['error' => 'invalid_provider'], 404);
        }
        $res = $client->debit($data);
        return response()->json($res['body'], $res['status']);
    }

    public function credit(Request $request, string $provider): JsonResponse
    {
        $request->merge([
            'player_id' => $request->input('player_id', $request->input('playerId')),
            'currency' => $request->input('currency', $request->input('currencyCode', env('GAMES_DEFAULT_CURRENCY', 'BD_BDT'))),
            'tx_id' => $request->input('tx_id', $request->input('txId')),
        ]);
        $data = $request->validate([
            'player_id' => 'required|string',
            'amount' => 'required|numeric',
            'currency' => 'required|string',
            'tx_id' => 'required|string',
        ]);
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->json(['error' => 'invalid_provider'], 404);
        }
        $res = $client->credit($data);
        return response()->json($res['body'], $res['status']);
    }

    public function rollback(Request $request, string $provider): JsonResponse
    {
        $data = $request->validate([
            'tx_id' => 'required|string',
        ]);
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->json(['error' => 'invalid_provider'], 404);
        }
        $res = $client->rollback($data);
        return response()->json($res['body'], $res['status']);
    }
}

