<?php

namespace App\Http\Controllers;

use App\Models\Game;
use App\Models\GameCategory;
use App\Services\Games\ProviderFactory;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;

class GameCatalogController extends Controller
{
    public function index(Request $request)
    {
        $playerId = null;
        if (auth()->check()) {
            $user = auth()->user();
            $player = $user->player ?? $user->player()->create([
                'user_id' => $user->id,
                'external_id' => 'USER-' . (string) $user->id,
                'status' => 'ACTIVE',
            ]);
            $playerId = $player->id;
        }
        $category = $request->query('category');
        $provider = 'agsoftgames';
        $client = ProviderFactory::make($provider);

        $items = [];
        $categories = collect();
        $debug = [];
        $refresh = (bool) $request->query('refresh');
        $existingQuery = Game::query()->where('active', true)->where('provider_key', $provider);
        if ($category) {
            $catModel = GameCategory::where('name', $category)->first();
            if ($catModel) { $existingQuery->where('category_id', $catModel->id); }
        }
        $existingCount = (int) $existingQuery->count();
        $minCount = (int) env('GAMES_SYNC_MIN_COUNT', 50);
        if ($client && ($refresh || $existingCount < $minCount)) {
            $catRes = $client->categories();
            $catBody = $catRes['body'] ?? null;
            if (is_array($catBody)) {
                $catData = $catBody['data'] ?? [];
                $catList = is_array($catData) ? ($catData['categoryDataList'] ?? $catData) : [];
                foreach ($catList as $c) {
                    $categories->push((string)($c['categoryCode'] ?? $c['categoryName'] ?? ''));
                }
                $categories = $categories->unique()->sort()->values();
            }

            $targetCats = $category ? [$category] : $categories->toArray();
            $itemsByCode = [];
            $attempts = [];
            $cfg = config('games.providers.' . $provider);
            $eps = (array)($cfg['endpoints'] ?? []);
            $logCount = 0;
            foreach ($targetCats as $catSel) {
                $page = 1;
                $pageSize = 50;
                $limit = 20;
                while ($limit-- > 0) {
                    $res = $client->games(['category' => $catSel, 'page' => $page, 'pageSize' => $pageSize]);
                    $body = $res['body'] ?? null;
                    $respData = is_array($body) ? ($body['data'] ?? $body) : [];
                    $list = is_array($respData) && isset($respData['gameItemList']) && is_array($respData['gameItemList']) ? $respData['gameItemList'] : (is_array($respData) ? $respData : []);
                    $attempts = array_merge($attempts, method_exists($client, 'attempts') ? $client->attempts() : []);
                    if (($res['status'] ?? 500) !== 200 || empty($list)) {
                        break;
                    }
                    foreach ($list as $it) {
                        $code = $it['gameCode'] ?? $it['code'] ?? $it['game_code'] ?? $it['id'] ?? null;
                        if (!$code || isset($itemsByCode[$code])) { continue; }
                        $name = $it['gameName'] ?? $it['name'] ?? $it['title'] ?? (string)$code;
                        $thumb = $it['thumbnail_url'] ?? $it['icon'] ?? $it['gameIcon'] ?? $it['iconUrl'] ?? $it['imageUrl'] ?? $it['coverUrl'] ?? $it['logo'] ?? null;
                        if (is_string($thumb) && $thumb !== '') {
                            if (str_starts_with($thumb, '//')) { $thumb = 'https:' . $thumb; }
                            if (str_starts_with($thumb, '/')) {
                                $base = (string) ($cfg['base_url'] ?? '');
                                $thumb = rtrim($base, '/') . $thumb;
                            }
                            if (!preg_match('/^https?:\\/\\//i', $thumb)) {
                                $base = (string) ($cfg['base_url'] ?? '');
                                $thumb = rtrim($base, '/') . '/' . ltrim($thumb, '/');
                            }
                        } else {
                            $iconBase = (string) env('GAMES_ICON_BASE', '');
                            if ($iconBase !== '') {
                                $thumb = rtrim($iconBase, '/') . '/' . (string)$code . '.jpg';
                            } else {
                                $thumb = 'https://ui-avatars.com/api/?name=' . urlencode((string)$name) . '&background=111&color=fff&size=512';
                            }
                        }
                        if ($logCount < 5) {
                            Log::channel('games')->info('catalog.item', [
                                'code' => $code,
                                'name' => $name,
                                'category' => $catSel,
                                'thumbnail_url' => $thumb,
                                'raw_keys' => array_keys($it),
                            ]);
                            $logCount++;
                        }
                        $itemsByCode[$code] = [
                            'provider_key' => $provider,
                            'code' => $code,
                            'name' => $name,
                            'category' => $catSel,
                            'thumbnail_url' => $thumb,
                            'status' => $it['status'] ?? null,
                            'game_type' => $it['gameType'] ?? null,
                        ];
                    }
                    if (count($list) < $pageSize) {
                        break;
                    }
                    $page++;
                }
            }
            $items = array_values($itemsByCode);
            foreach ($items as $it) {
                $catName = $it['category'] ?: 'General';
                $catModel = GameCategory::firstOrCreate(['name' => $catName]);
                \App\Models\Game::updateOrCreate(
                    ['code' => (string) $it['code']],
                    [
                        'provider_key' => (string) $it['provider_key'],
                        'name' => (string) $it['name'],
                        'category_id' => $catModel->id,
                        'thumbnail_url' => $it['thumbnail_url'] ?? null,
                        'meta' => [
                            'status' => $it['status'] ?? null,
                            'game_type' => $it['game_type'] ?? null,
                        ],
                        'active' => true,
                    ]
                );
            }
            $items = [];
            $debug = [
                'provider' => $provider,
                'base_url' => (string)($cfg['base_url'] ?? ''),
                'primary_path' => (string)($eps['games'] ?? ''),
                'candidates' => (array)($eps['games_candidates'] ?? []),
                'status' => 200,
                'attempts' => $attempts,
                'count' => count($items),
                'reason' => null,
                'body' => null,
                'headers' => [],
            ];
        }

        if (empty($items)) {
            $query = Game::query()->where('active', true)->where('provider_key', $provider);
            $q = trim((string) $request->query('q', ''));
            $hasImage = in_array(strtolower((string) $request->query('has_image')), ['1','true','yes','on']);
            $gameType = trim((string) $request->query('game_type', ''));
            if ($category) {
                $cat = GameCategory::where('name', $category)->first();
                if ($cat) {
                    $query->where('category_id', $cat->id);
                }
            }
            if ($q !== '') {
                $query->where('name', 'like', '%' . str_replace(['%','_'], ['\\%','\\_'], $q) . '%');
            }
            if ($hasImage) {
                $query->whereNotNull('thumbnail_url');
            }
            if ($gameType !== '') {
                $query->where('meta->game_type', $gameType);
            }
            $games = $query->orderBy('name')->paginate(12);
            $categories = GameCategory::whereHas('games', function ($q) use ($provider) {
                $q->where('provider_key', $provider)->where('active', true);
            })->orderBy('name')->pluck('name');
            $categoryLabels = $categories->mapWithKeys(function ($n) {
                $name = is_object($n) ? ($n->name ?? '') : (string) $n;
                $label = preg_replace('/^AG_(HOT_)?/', '', $name);
                return [$name => $label];
            });
            $gameTypes = Game::query()
                ->where('active', true)
                ->where('provider_key', $provider)
                ->pluck('meta')
                ->map(function ($m) {
                    if (is_array($m)) {
                        return $m['game_type'] ?? null;
                    }
                    if (is_string($m)) {
                        $decoded = json_decode($m, true);
                        return is_array($decoded) ? ($decoded['game_type'] ?? null) : null;
                    }
                    return null;
                })
                ->filter()
                ->unique()
                ->sort()
                ->values();
            return view('games.index', compact('games', 'categories', 'category', 'debug', 'gameTypes', 'categoryLabels', 'playerId'))->with(['q' => $q, 'hasImage' => $hasImage, 'gameType' => $gameType]);
        }

        $page = (int)$request->query('page', 1);
        $perPage = 12;
        $total = count($items);
        $slice = array_slice($items, ($page - 1) * $perPage, $perPage);
        $slice = array_map(function ($it) { return (object) $it; }, $slice);
        $games = new LengthAwarePaginator($slice, $total, $perPage, $page, [
            'path' => $request->url(),
            'query' => $request->query(),
        ]);
        if ($request->expectsJson() || $request->query('format') === 'json') {
            return response()->json([
                'provider' => $debug['provider'] ?? null,
                'base_url' => $debug['base_url'] ?? null,
                'primary_path' => $debug['primary_path'] ?? null,
                'candidates' => $debug['candidates'] ?? [],
                'status' => $debug['status'] ?? null,
                'reason' => $debug['reason'] ?? null,
                'headers' => $debug['headers'] ?? [],
                'response_body' => $debug['body'] ?? null,
                'count' => $total,
                'items' => $slice,
            ]);
        }

        return view('games.index', compact('games', 'categories', 'category', 'debug', 'playerId'));
    }
}
