<?php

namespace App\Http\Controllers;

use App\Services\Games\ProviderFactory;
use App\Models\CallbackEvent;
use App\Models\GameTransaction;
use App\Models\Wallet;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\JsonResponse;

class GameCallbackController extends Controller
{
    public function handle(Request $request, string $provider): JsonResponse
    {
        Log::channel('games')->info('games.callback.request', [
            'provider' => $provider,
            'method' => $request->method(),
            'ip' => $request->ip(),
            'query' => $request->query(),
            'headers' => $request->headers->all(),
            'raw' => $request->getContent(),
            'json' => $request->json()->all(),
            'event' => 'games_callback_request',
        ]);
        $client = ProviderFactory::make($provider);
        if (!$client) {
            return response()->json(['error' => 'invalid_provider'], 404);
        }
        $payload = $request->getContent();
        $signature = $request->header('X-SIGN') ?? $request->header('X-Signature') ?? $request->header('X-SIGNATURE');
        $valid = $client->verifySignature($payload, $signature, $request->headers->all());
        Log::channel('games')->info('games.callback', [
            'provider' => $provider,
            'valid' => $valid,
            'event' => 'games_callback_signature_validated',
        ]);
        if (!$valid) {
            return response()->json(['error' => 'invalid_signature'], 401);
        }
        $json = $request->json()->all();
        CallbackEvent::create([
            'provider_key' => $provider,
            'event_type' => (string)($json['event'] ?? ''),
            'signature_valid' => true,
            'payload' => $payload,
            'handled_at' => now(),
        ]);

        $txId = (string)($json['tx_id'] ?? '');
        if ($txId === '') {
            return response()->json(['status' => 'ok']);
        }

        $existing = GameTransaction::where('tx_id', $txId)->first();
        if ($existing && $existing->status === 'processed') {
            return response()->json(['status' => 'ok']);
        }

        DB::transaction(function () use ($json, $provider, $txId) {
            $type = (string)($json['type'] ?? '');
            $amount = (float)($json['amount'] ?? 0);
            $currency = (string)($json['currency'] ?? '');
            $playerId = (string)($json['player_id'] ?? '');
            $sessionId = (string)($json['session_id'] ?? '');

            $wallet = Wallet::firstOrCreate([
                'player_id' => $playerId,
                'currency' => $currency,
            ]);

            if ($type === 'debit') {
                $wallet->balance = $wallet->balance - $amount;
            } elseif ($type === 'credit') {
                $wallet->balance = $wallet->balance + $amount;
            }
            $wallet->save();

            GameTransaction::updateOrCreate(
                ['tx_id' => $txId],
                [
                    'provider_key' => $provider,
                    'session_id' => $sessionId,
                    'type' => $type,
                    'amount' => $amount,
                    'currency' => $currency,
                    'status' => 'processed',
                    'processed_at' => now(),
                ]
            );
        });

        return response()->json(['status' => 'ok']);
    }
}
