<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Wallet;
use App\Models\GameTransaction;
use App\Models\GameSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;

class AdminController extends Controller
{
    public function index()
    {
        return View::make('admin.dashboard');
    }

    public function users()
    {
        $users = User::orderBy('name')->get();
        $balances = Wallet::select('player_id', DB::raw('SUM(balance) as total_balance'))
            ->groupBy('player_id')->pluck('total_balance', 'player_id');
        return View::make('admin.users', compact('users', 'balances'));
    }

    public function recharge(Request $request, User $user)
    {
        $data = $request->validate(['amount' => 'required|numeric|min:0.01', 'currency' => 'required|string']);
        DB::transaction(function () use ($user, $data) {
            $wallet = Wallet::lockForUpdate()->firstOrCreate(['player_id' => $user->id, 'currency' => $data['currency']]);
            $wallet->balance = $wallet->balance + $data['amount'];
            $wallet->save();
        });
        return redirect('/admin/users')->with('status', 'recharged');
    }

    public function transactions(Request $request)
    {
        $playerId = (int) $request->query('player_id', 0);
        $query = GameTransaction::query()
            ->leftJoin('game_sessions', 'game_transactions.session_id', '=', 'game_sessions.session_id')
            ->select(['game_transactions.*', 'game_sessions.player_id']);
        if ($playerId > 0) {
            $query->where('game_sessions.player_id', $playerId);
        }
        $tx = $query->orderByDesc('game_transactions.id')->limit(200)->get();
        $agg = GameTransaction::query()
            ->leftJoin('game_sessions', 'game_transactions.session_id', '=', 'game_sessions.session_id')
            ->select([
                'game_sessions.player_id',
                DB::raw("SUM(CASE WHEN game_transactions.type='debit' THEN game_transactions.amount ELSE 0 END) AS total_debit"),
                DB::raw("SUM(CASE WHEN game_transactions.type='credit' THEN game_transactions.amount ELSE 0 END) AS total_credit"),
            ])
            ->groupBy('game_sessions.player_id')
            ->get()
            ->map(function ($row) {
                $row->net = (float)($row->total_credit ?? 0) - (float)($row->total_debit ?? 0);
                return $row;
            });
        return View::make('admin.transactions', ['transactions' => $tx, 'aggregate' => $agg, 'playerId' => $playerId]);
    }
}

